<?php
// DuckChatClient.php
// نصب شده: composer require chrome-php/chrome
// نیاز: Chromium/Chrome نصب شده و مسیر آن را مشخص کنید.

require __DIR__ . '/vendor/autoload.php';

use HeadlessChromium\BrowserFactory;

class DuckChatClient
{
    private $chromePath;
    private $userAgent;
    private $timeoutMs;
    private $noSandbox;

    /**
     * @param string|null $chromePath مسیر اجرایی کروم/کرومیوم، اگر null از پیش‌فرض استفاده می‌شود
     * @param array $opts تنظیمات اضافی: 'userAgent', 'timeoutMs', 'noSandbox'
     */
    public function __construct(string $chromePath = null, array $opts = [])
    {
        $this->chromePath = $chromePath;
        $this->userAgent = $opts['userAgent'] ?? 'Mozilla/5.0 (Linux; Android 10; K) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/137.0.0.0 Mobile Safari/537.36';
        $this->timeoutMs = $opts['timeoutMs'] ?? 120000; // زمان پیش‌فرض 120s
        $this->noSandbox = $opts['noSandbox'] ?? true;
    }

    /**
     * ارسال پرسش و دریافت پاسخ کامل (stream جمع‌آوری می‌شود).
     * برمی‌گرداند آرایه ['ok' => true, 'status' => int, 'text' => string] یا ['ok'=>false,'error'=>string]
     */
    public function ask(string $prompt): array
    {
        $browserFactory = $this->chromePath ? new BrowserFactory($this->chromePath) : new BrowserFactory();

        // هر درخواست یک userDataDir مجزا (برای جلوگیری از conflict)
        $userDataDir = sys_get_temp_dir() . '/duckchat_profile_' . bin2hex(random_bytes(6));

        $startOptions = [
            'headless' => true,
            'noSandbox' => $this->noSandbox,
            'userDataDir' => $userDataDir,
            // می‌تونید منابع رو محدود کنید اگر لازم بود
            // 'connectionDelay' => 0,
        ];

        $browser = null;
        try {
            $browser = $browserFactory->createBrowser($startOptions);
            $page = $browser->createPage();

            // ست کردن UA
            $page->setUserAgent($this->userAgent);

            // باز کردن صفحه و صبر برای لود
            $page->navigate('https://duckduckgo.com')->waitForNavigation();

            // JS که در صفحه اجرا می‌شود: fetch به /duckchat/v1/chat و خواندن stream درون مرورگر
            $js = <<<'JS'
(async function(prompt) {
  const payload = {
    model: "gpt-4o-mini",
    metadata: { toolChoice: { NewsSearch:false, VideosSearch:false, LocalSearch:false, WeatherForecast:false } },
    messages: [{ role: "user", content: prompt }],
    canUseTools: true,
    canUseApproxLocation: false
  };
  try {
    const res = await fetch('/duckchat/v1/chat', {
      method: 'POST',
      headers: {
        'accept': 'text/event-stream',
        'content-type': 'application/json'
      },
      body: JSON.stringify(payload),
      credentials: 'include'
    });

    // اگر stream است، reader را می‌خوانیم تکه‌تکه
    let text = '';
    try {
      const reader = res.body?.getReader?.();
      if (reader) {
        const decoder = new TextDecoder();
        while (true) {
          const { done, value } = await reader.read();
          if (done) break;
          text += decoder.decode(value || new Uint8Array(), { stream: true });
          // برای جلوگیری از حافظه زیاد می‌توانیم trim کنیم؛ اینجا کامل جمع می‌کنیم
        }
      } else {
        text = await res.text();
      }
    } catch (e) {
      // خواندن stream ممکنه خطا بده؛ در این صورت متنی از خطا برمی‌گردونیم
      return { status: res.status, ok: false, error: 'stream-read-error: ' + String(e) };
    }
    return { status: res.status, ok: res.ok, text };
  } catch (err) {
    return { status: 0, ok: false, error: String(err) };
  }
})
JS;

            // اجرا و دریافت نتیجه
            // توجه: evaluate با آرگومان‌ها پشتیبانی می‌شود
            $evaluation = $page->evaluate($js, $prompt);
            // صبر برای نتیجه با timeout
            $wait = $evaluation->getReturnValue(); // chrome-php همزمان صبر می‌کند

            // ممکن است مقدار خیلی بزرگ باشد؛ اینجا دریافت می‌کنیم
            if (!is_array($wait)) {
                // اگر فقط رشته برگشت داده شد
                return ['ok' => true, 'status' => 200, 'text' => (string)$wait];
            }

            return $wait;
        } catch (\Throwable $e) {
            return ['ok' => false, 'error' => $e->getMessage()];
        } finally {
            if ($browser) {
                try { $browser->close(); } catch (\Throwable $e) { /* ignore */ }
            }
            // حذف userDataDir در صورت وجود
            if (isset($userDataDir) && is_dir($userDataDir)) {
                $this->rrmdir($userDataDir);
            }
        }
    }

    // حذف دایرکتوری و زیرشاخه‌ها
    private function rrmdir($dir) {
        if (!is_dir($dir)) return;
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ($files as $fileinfo) {
            $todo = ($fileinfo->isDir() ? 'rmdir' : 'unlink');
            @$todo($fileinfo->getRealPath());
        }
        @rmdir($dir);
    }
}